//	TorusGamesOptions.c
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "TorusGames-Common.h"
#include "GeometryGamesUtilities-Common.h"


//	In a 3D game, when changing to ViewRepeating,
//	how long should the snap-to-axes take?
#define CHANGE_VIEW_AXIS_SNAP_DURATION		0.25	//	in seconds


void SetShowGlideAxes(
	ModelData	*md,
	bool		aShowGlideAxesChoice)
{
	md->itsShowGlideAxes = aShowGlideAxesChoice;
	md->itsChangeCount++;
}


bool GameIs3D(
	GameType	aGame)
{
	switch (aGame)
	{
		case GameNone:		//	treat as 2D
		case Game2DIntro:
		case Game2DTicTacToe:
		case Game2DGomoku:
		case Game2DMaze:
		case Game2DCrossword:
		case Game2DWordSearch:
		case Game2DJigsaw:
		case Game2DChess:
		case Game2DPool:
		case Game2DApples:
			return false;
		
		case Game3DTicTacToe:
		case Game3DMaze:
			return true;
		
		default:
			GeometryGamesFatalError(u"GameIs3D() received an unknown GameType.", u"Internal Error");
			return false;	//	suppress compiler warnings
	}
}

bool TopologyIs3D(
	TopologyType	aTopology)
{
	switch (aTopology)
	{
		case Topology2DTorus:
		case Topology2DKlein:
			return false;
		
		case Topology3DTorus:
		case Topology3DKlein:
		case Topology3DQuarterTurn:
		case Topology3DHalfTurn:
			return true;
		
		default:
			GeometryGamesFatalError(u"TopologyIs3D() received an unknown TopologyType.", u"Internal Error");
			return false;	//	suppress compiler warnings
	}
}


void ChangeHumanVsComputer(
	ModelData	*md,
	bool		aNewHumanVsComputer)
{
	if (md->itsHumanVsComputer != aNewHumanVsComputer)
	{
		md->itsHumanVsComputer = aNewHumanVsComputer;

		if (md->itsGameHumanVsComputerChanged != NULL)
			(*md->itsGameHumanVsComputerChanged)(md);	//	let Pool assign computer player

		RefreshStatusMessageText(md);	//	necessary only for Pool

		md->itsChangeCount++;
	}
}


void ChangeDifficultyLevel(
	ModelData		*md,
	unsigned int	aNewDifficultyLevel)
{
	//	Just to be safe...
	if (aNewDifficultyLevel > 3)
		return;

	//	Ignore redundant changes to prevent unneeded resets.
	if (aNewDifficultyLevel == md->itsDifficultyLevel)
		return;

	//	Set the new difficulty level.
	md->itsDifficultyLevel = aNewDifficultyLevel;

	//	The maze, jigsaw puzzle and apples games need to reset,
	//	but gomoku and chess do not.
	if (md->itsGame == Game2DMaze
	 || md->itsGame == Game2DJigsaw
	 || md->itsGame == Game2DApples
	 || md->itsGame == Game3DMaze)
		ResetGame(md);
}


void ChangeTopology(
	ModelData		*md,
	TopologyType	aNewTopology)
{
	GEOMETRY_GAMES_ASSERT(
		aNewTopology < NumTopologyTypes,
		"Unknown TopologyType");

	//	The user never explicitly changes
	//	from a 2D topology to a 3D topology.
	//	The dimensionality changes only when
	//	the user moves from a 2D game to a 3D game.
	GEOMETRY_GAMES_ASSERT(
		TopologyIs3D(aNewTopology) == TopologyIs3D(md->itsTopology),
		"New and old topology have different dimensions.");

	//	Ignore redundant changes to prevent unneeded resets.
	if (aNewTopology == md->itsTopology)
		return;

	//	Set the new topology.
	md->itsTopology = aNewTopology;

	//	Reset the game.
	ResetGame(md);

	//	Request a redraw.
	md->itsChangeCount++;
}


void ChangeViewType(
	ModelData	*md,
	ViewType	aNewViewType)
{
	//	Just to be safe...
	if (aNewViewType != ViewBasicLarge
	 && aNewViewType != ViewBasicSmall
	 && aNewViewType != ViewRepeating)
		return;

	//	Ignore redundant changes.
	if (aNewViewType == md->itsViewType)
		return;

	//	If a simulation is already running,
	//	ignore this call to ChangeViewType().
	//	We'll need to start a simulation of our own.
	if (md->itsSimulationStatus != SimulationNone)
		return;
	
	//	Animate the view transition.
	if (GameIs3D(md->itsGame))
	{
		switch (aNewViewType)
		{
			case ViewBasicLarge:

				md->its3DDesiredAperture = 0.0;
				SimulationBegin(md, Simulation3DExitTilingMode);

				//	If the user later changes to a 2D game,
				//	the 2D view magnification factor should be 1.
				md->its2DViewMagFactor			= 1.0;
				md->itsDesired2DViewMagFactor	= 1.0;

				break;
		
			case ViewRepeating:

				//	SnapToAxes3D(…, 0.0, …, true) will call
				//
				//		SimulationBegin(md, Simulation3DRotationSnapToAxes)
				//
				//	which, upon completion, will set
				//
				//		md->itsViewType				= ViewRepeating;
				//		md->its3DDesiredAperture	= 1.0;
				//
				SnapToAxes3D(md, 0.0, CHANGE_VIEW_AXIS_SNAP_DURATION, true);

				//	If the user later changes to a 2D game,
				//	the 2D view magnification factor should be 1/3.
				md->its2DViewMagFactor			= 1.0 / 3.0;
				md->itsDesired2DViewMagFactor	= 1.0 / 3.0;

				break;
			
			default:
				GEOMETRY_GAMES_ABORT("Unexpected new ViewType");
				break;
		}
	}
	else	//	2D game
	{
		switch (aNewViewType)
		{
			case ViewBasicLarge:
				switch (md->itsViewType)
				{
					case ViewBasicLarge:
						//	should never occur
						break;
					
					case ViewBasicSmall:
						md->itsViewType					= ViewBasicSmall;	//	redundant
						md->itsDesiredViewType			= ViewBasicLarge;
						md->itsDesired2DViewMagFactor	= 1.0;
						break;
					
					case ViewRepeating:
						md->itsViewType					= ViewBasicSmall;	//	not redundant
						md->itsDesiredViewType			= ViewBasicLarge;
						md->itsDesired2DViewMagFactor	= 1.0;
						break;
				}
				break;
			
			case ViewBasicSmall:
				switch (md->itsViewType)
				{
					case ViewBasicLarge:
						md->itsViewType					= ViewBasicSmall;
						md->itsDesiredViewType			= ViewBasicSmall;
						md->itsDesired2DViewMagFactor	= 1.0 / 3.0;
						break;
					
					case ViewBasicSmall:
						//	should never occur
						break;
					
					case ViewRepeating:
						md->itsViewType					= ViewBasicSmall;
						md->itsDesiredViewType			= ViewBasicSmall;
						md->itsDesired2DViewMagFactor	= 1.0 / 3.0;
						break;
				}
				break;
		
			case ViewRepeating:
				switch (md->itsViewType)
				{
					case ViewBasicLarge:
						md->itsViewType					= ViewBasicSmall;
						md->itsDesiredViewType			= ViewRepeating;
						md->itsDesired2DViewMagFactor	= 1.0 / 3.0;
						break;
					
					case ViewBasicSmall:
						md->itsViewType					= ViewRepeating;
						md->itsDesiredViewType			= ViewRepeating;
						md->itsDesired2DViewMagFactor	= 1.0 / 3.0;
						break;
					
					case ViewRepeating:
						//	should never occur
						break;
				}
				break;
			
			default:
				GEOMETRY_GAMES_ABORT("Unexpected new ViewType");
				break;
		}
	}

	//	Request a redraw.
	md->itsChangeCount++;
}


void SetNewGame(
	ModelData	*md,
	GameType	aNewGame)
{
	//	The platform-specific code implements a Change Game function,
	//	because it needs to update UI elements as well as changing
	//	the game itself.  Here we merely
	//
	//		set the requested game,
	//		make sure the TopologyType is suitable for the new game, and
	//		make sure the     ViewType is suitable for the new game.
	//
	
	//	Caution:  This function assumes the caller
	//	has already shut down the old game,
	//	and will soon set up the new game.

	//	Switch to the new game.
	md->itsGame = aNewGame;

	//	Make sure the TopologyType is
	//	2D for a 2D game or 3D for a 3D Game.
	switch (md->itsGame)
	{
		case GameNone:	//	treat as 2D
		case Game2DIntro:
		case Game2DTicTacToe:
		case Game2DGomoku:
		case Game2DMaze:
		case Game2DCrossword:
		case Game2DWordSearch:
		case Game2DJigsaw:
		case Game2DChess:
		case Game2DPool:
		case Game2DApples:
		
			//	Switch from a 3D topology to a 2D topology, if necessary.
			switch (md->itsTopology)
			{
				case Topology2DTorus:
				case Topology2DKlein:
					//	Topology is already 2D.
					//	No change is needed.
					break;
		
				case Topology3DTorus:
				case Topology3DQuarterTurn:
				case Topology3DHalfTurn:
					md->itsTopology = Topology2DTorus;
					break;

				case Topology3DKlein:
					md->itsTopology = Topology2DKlein;
					break;
				
				default:
					GeometryGamesFatalError(u"SetNewGame() received an unknown topology.", u"Internal Error");
					break;
			}

			break;

		case Game3DTicTacToe:
		case Game3DMaze:

			//	Switch from a 2D topology to a 3D topology, if necessary.
			switch (md->itsTopology)
			{
				case Topology2DTorus:
					md->itsTopology = Topology3DTorus;
					break;
		
				case Topology2DKlein:
					md->itsTopology = Topology3DKlein;
					break;
		
				case Topology3DTorus:
				case Topology3DKlein:
				case Topology3DQuarterTurn:
				case Topology3DHalfTurn:
					//	Topology is already 3D.
					//	No change is needed.
					break;
				
				default:
					GeometryGamesFatalError(u"SetNewGame() received an unknown topology.", u"Internal Error");
					break;
			}
			
			//	3D games don't need ViewBasicSmall and don't support it,
			//	so map ViewBasicSmall to ViewBasicLarge.
			if (md->itsViewType == ViewBasicSmall)
				md->itsViewType = ViewBasicLarge;
			
			//	Make sure the aperture is consistent with the ViewType.
			switch (md->itsViewType)
			{
				case ViewBasicLarge:
					md->its3DDesiredAperture = 0.0;
					md->its3DCurrentAperture = 0.0;
					break;

				case ViewRepeating:
					md->its3DDesiredAperture = 1.0;
					md->its3DCurrentAperture = 1.0;
					break;
				
				default:	//	should never occur
					break;
			}

			break;
		
		default:
			GeometryGamesFatalError(u"SetNewGame() received an unknown game.", u"Internal Error");
			break;
	}
}
